# ===== 新增：日志管理模块 =====
import os
from urllib.parse import quote

from PyQt5.QtCore import Qt, pyqtSignal, QObject
from PyQt5.QtWidgets import (QApplication, QFileDialog,
                             QMessageBox, QInputDialog, QLineEdit, QProgressDialog, QListWidget)

from yc import LogManager


class FileManager(QObject):
    """负责文件上传下载、重命名删除等文件操作的管理类"""
    # 定义信号，用于线程间通信
    upload_progress_signal = pyqtSignal(int, int)  # 上传进度信号
    download_progress_signal = pyqtSignal(int, int)  # 下载进度信号
    show_temp_status_message = pyqtSignal(str)  # 临时状态消息信号

    def __init__(self, main_window):
        """
        初始化文件管理器
        :param main_window: 主窗口实例，用于访问主窗口的属性和方法
        """
        super().__init__()
        self.main_window = main_window
        self.logger = LogManager()
        self.threads = []  # 管理后台线程
        # 连接进度信号到主窗口的处理方法
        self.upload_progress_signal.connect(self._on_upload_progress)
        self.download_progress_signal.connect(self._on_download_progress)
        self.show_temp_status_message.connect(self._on_show_temp_status_message)

    # ===== 属性访问器 =====
    @property
    def webdav_url(self):
        """获取WebDAV URL（只读）"""
        return self.main_window.webdav_url

    @property
    def username(self):
        """获取用户名（只读）"""
        return self.main_window.username

    @property
    def password(self):
        """获取密码（只读）"""
        return self.main_window.password

    @property
    def track_list(self):
        """获取音乐文件列表（只读）"""
        return self.main_window.track_list

    @property
    def playlist_manager(self):
        """获取播放列表管理器（只读）"""
        return self.main_window.playlist_manager

    @property
    def log_manager(self):
        """获取日志管理器（只读）"""
        return self.main_window.log_manager

    def get_session(self):
        """获取HTTP会话对象（通过主窗口的方法）"""
        return self.main_window.get_session()

    # ===== 文件操作方法 =====
    def upload_files(self):
        """上传文件到WebDAV服务器"""
        self.logger.debug("上传文件到WebDAV服务器")
        try:
            # 选择文件
            file_paths, _ = QFileDialog.getOpenFileNames(
                self.main_window, "选择要上传的文件", "", "所有文件 (*)"
            )
            if not file_paths:
                self.logger.debug("用户取消了文件选择")
                return
            self.logger.debug(f"用户选择了 {len(file_paths)} 个文件上传")
            # 调用通用上传方法
            self._upload_files_with_paths(file_paths)
        except Exception as e:
            self.logger.debug(f"上传文件时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.main_window, "错误", f"上传文件失败: {str(e)}")

    def create_drag_drop_list(self, parent=None):
        """工厂方法：创建拖拽列表组件"""
        return DragDropListWidget(parent, self.main_window, self)

    def _upload_files_with_paths(self, file_paths):
        """通用文件上传方法，接受文件路径列表"""
        self.logger.debug(f"开始上传 {len(file_paths)} 个文件")
        try:
            # 创建进度对话框
            progress_dialog = QProgressDialog("正在上传文件...", "取消", 0, len(file_paths), self.main_window)
            progress_dialog.setWindowModality(Qt.WindowModal)
            progress_dialog.setWindowTitle("上传进度")
            progress_dialog.setMinimumDuration(0)
            progress_dialog.setValue(0)
            # 上传文件
            success_count = 0
            failure_count = 0
            for i, file_path in enumerate(file_paths):
                if progress_dialog.wasCanceled():
                    self.logger.debug("用户取消了上传")
                    break
                file_name = os.path.basename(file_path)
                progress_dialog.setLabelText(f"正在上传: {file_name}")
                progress_dialog.setValue(i)
                try:
                    # 上传文件
                    self._upload_single_file(file_path)
                    success_count += 1
                    self.logger.info(f"文件上传成功: {file_name}")
                except Exception as e:
                    failure_count += 1
                    self.logger.debug(f"文件上传失败: {file_name}, 错误: {e}")
                # 更新进度
                self.upload_progress_signal.emit(i + 1, len(file_paths))
                QApplication.processEvents()  # 保持UI响应
            progress_dialog.setValue(len(file_paths))
            # 显示结果
            if failure_count == 0:
                self.show_temp_status_message.emit(f"成功上传 {success_count} 个文件")
            else:
                self.show_temp_status_message.emit(f"上传完成: 成功 {success_count} 个, 失败 {failure_count} 个")
            # 刷新文件列表
            self.main_window.connect_and_load()
        except Exception as e:
            self.logger.debug(f"上传文件时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.main_window, "错误", f"上传文件失败: {str(e)}")

    def _upload_single_file(self, file_path):
        """上传单个文件到WebDAV服务器"""
        self.logger.debug(f"上传单个文件: {file_path}")
        # 获取文件名
        file_name = os.path.basename(file_path)
        # 对文件名进行URL编码
        encoded_file_name = quote(file_name)
        # 构建上传路径（直接使用文件名，上传到WebDAV根目录）
        upload_path = f"/{encoded_file_name}"
        # 构建上传URL（不包含认证信息，使用session）
        upload_url = self.playlist_manager.build_webdav_url(upload_path, include_auth=False)
        self.logger.debug(f"上传URL: {upload_url}")
        # 确保会话已初始化
        session = self.get_session()
        # 读取文件内容
        with open(file_path, 'rb') as f:
            file_data = f.read()
        # 上传文件（使用session的认证信息）
        headers = {
            'Content-Type': 'application/octet-stream'
        }
        response = session.put(upload_url, data=file_data, headers=headers, timeout=30)
        if response.status_code not in (200, 201, 204):
            raise Exception(f"上传失败，服务器返回状态码: {response.status_code}")
        self.logger.debug(f"文件上传成功: {file_name}")

    def download_selected_files(self):
        """下载选中的文件"""
        self.logger.debug("下载选中的文件")
        try:
            selected_items = self.main_window.file_list.selectedItems()
            if not selected_items:
                self.logger.debug("没有选中文件")
                return
            # 提前复制所有选中项的数据，避免在遍历过程中项被删除
            selected_data = []
            for item in selected_items:
                file_name = item.data(Qt.UserRole + 1)
                file_path = item.data(Qt.UserRole)
                selected_data.append((file_name, file_path))
            # 选择保存目录
            save_dir = QFileDialog.getExistingDirectory(self.main_window, "选择保存目录")
            if not save_dir:
                self.logger.debug("用户取消了目录选择")
                return
            self.logger.debug(f"用户选择保存目录: {save_dir}")
            # 创建进度对话框
            progress_dialog = QProgressDialog("正在下载文件...", "取消", 0, len(selected_data), self.main_window)
            progress_dialog.setWindowModality(Qt.WindowModal)
            progress_dialog.setWindowTitle("下载进度")
            progress_dialog.setMinimumDuration(0)
            progress_dialog.setValue(0)
            # 下载文件
            success_count = 0
            failure_count = 0
            for i, (file_name, file_path) in enumerate(selected_data):
                if progress_dialog.wasCanceled():
                    self.logger.debug("用户取消了下载")
                    break
                progress_dialog.setLabelText(f"正在下载: {file_name}")
                progress_dialog.setValue(i)
                try:
                    # 下载文件
                    save_path = os.path.join(save_dir, file_name)
                    self.main_window._download_single_file(file_path, save_path)
                    success_count += 1
                    self.logger.debug(f"文件下载成功: {file_name}")
                except Exception as e:
                    failure_count += 1
                    self.logger.debug(f"文件下载失败: {file_name}, 错误: {e}")
                # 更新进度
                self.download_progress_signal.emit(i + 1, len(selected_data))
                QApplication.processEvents()  # 保持UI响应
            progress_dialog.setValue(len(selected_data))
            # 显示结果
            if failure_count == 0:
                self.show_temp_status_message.emit(f"成功下载 {success_count} 个文件")
            else:
                self.show_temp_status_message.emit(f"下载完成: 成功 {success_count} 个, 失败 {failure_count} 个")
        except Exception as e:
            self.logger.debug(f"下载文件时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.main_window, "错误", f"下载文件失败: {str(e)}")

    def rename_selected_files(self):
        """重命名选中的文件"""
        self.logger.debug("重命名选中的文件")
        try:
            selected_items = self.main_window.file_list.selectedItems()
            if not selected_items:
                self.logger.debug("没有选中文件")
                return
            if len(selected_items) > 1:
                self.logger.debug("选中了多个文件，只重命名第一个")
                selected_items = [selected_items[0]]
            # 获取文件信息
            item = selected_items[0]
            old_file_name = item.data(Qt.UserRole + 1)
            old_file_path = item.data(Qt.UserRole)
            self.logger.debug(f"重命名文件: {old_file_name}")
            # 输入新文件名
            new_file_name, ok = QInputDialog.getText(
                self.main_window, "重命名文件",
                f"请输入新的文件名（保留扩展名）:",
                QLineEdit.Normal,
                old_file_name
            )
            if not ok or not new_file_name:
                self.logger.debug("用户取消了重命名")
                return
            # 检查文件名是否相同
            if new_file_name == old_file_name:
                self.logger.debug("文件名未改变")
                return
            # 获取文件扩展名
            old_ext = os.path.splitext(old_file_name)[1]
            new_ext = os.path.splitext(new_file_name)[1]
            # 如果新文件名没有扩展名，使用旧扩展名
            if not new_ext and old_ext:
                new_file_name += old_ext
                self.logger.debug(f"添加扩展名: {new_file_name}")
            self.logger.debug(f"新文件名: {new_file_name}")
            # 执行重命名
            self._rename_single_file(old_file_path, new_file_name)
            # 显示成功消息
            self.show_temp_status_message.emit(f"文件已重命名为: {new_file_name}")
            # 刷新文件列表
            self.main_window.connect_and_load()
        except Exception as e:
            self.logger.debug(f"重命名文件时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.main_window, "错误", f"重命名文件失败: {str(e)}")

    def _rename_single_file(self, webdav_path, new_file_name):
        """重命名单个文件"""
        self.logger.info(f"重命名单个文件: {webdav_path} -> {new_file_name}")
        try:
            # 使用原始版本的简单URL构建方式
            protocol = self.webdav_url.split('://')[0]
            host = self.webdav_url.split('://')[1].split('/')[0]
            base_webdav = f"{protocol}://{host}"

            # 获取文件所在目录
            dir_path = os.path.dirname(webdav_path)
            if not dir_path.endswith('/'):
                dir_path += '/'

            # 构建源URL（直接使用原始路径）
            source_url = f"{base_webdav}{webdav_path}"

            # 关键修复：对整个目标URL进行IRI（国际化资源标识符）编码
            # 而不仅仅是文件名
            new_file_path = os.path.join(dir_path, new_file_name)

            # 使用QUrl来正确处理包含非ASCII字符的URL
            from PyQt5.QtCore import QUrl

            # 构建完整的目标URL
            full_dest_url = f"{base_webdav}{new_file_path}"

            # 使用QUrl进行编码，它会自动处理非ASCII字符
            qurl = QUrl(full_dest_url)
            destination_url = qurl.toString()

            self.logger.debug(f"原路径: {webdav_path}")
            self.logger.debug(f"新路径: {new_file_path}")
            self.logger.debug(f"源URL: {source_url}")
            self.logger.debug(f"目标URL: {destination_url}")

            # 确保会话已初始化
            session = self.get_session()
            if session is None:
                raise Exception("无法获取HTTP会话")

            # 关键修复：修改Overwrite为'T'，确保覆盖目标文件
            headers = {
                'Destination': destination_url,
                'Overwrite': 'T'  # 修改为'T'，确保覆盖目标文件
            }

            response = session.request('MOVE', source_url, headers=headers, timeout=30)

            if response.status_code not in (200, 201, 204):
                error_msg = f"重命名失败，服务器返回状态码: {response.status_code}"
                if response.text:
                    error_msg += f", 响应内容: {response.text}"
                self.logger.debug(error_msg)
                raise Exception(error_msg)

            self.logger.info(f"文件重命名成功: {webdav_path} -> {new_file_name}")

        except UnicodeEncodeError as e:
            self.logger.error(f"URL编码错误: {e}")
            # 尝试使用更激进的编码方式作为备用方案
            try:
                # 重新构建URL，但使用手动编码方式
                protocol = self.webdav_url.split('://')[0]
                host = self.webdav_url.split('://')[1].split('/')[0]
                base_webdav = f"{protocol}://{host}"

                # 获取文件所在目录
                dir_path = os.path.dirname(webdav_path)
                if not dir_path.endswith('/'):
                    dir_path += '/'

                # 构建源URL
                source_url = f"{base_webdav}{webdav_path}"

                # 手动编码路径的每个部分
                from urllib.parse import quote

                def iri_encode_path(path):
                    """将路径编码为IRI格式"""
                    # 分割路径为各个部分
                    parts = path.strip('/').split('/')
                    # 对每个部分进行UTF-8编码后URL编码
                    encoded_parts = []
                    for part in parts:
                        # 使用UTF-8编码后进行URL编码
                        encoded_part = quote(part.encode('utf-8'), safe='')
                        encoded_parts.append(encoded_part)
                    # 重新组合路径
                    return '/' + '/'.join(encoded_parts)

                # 构建新文件完整路径并编码
                new_file_path = os.path.join(dir_path, new_file_name)
                encoded_new_path = iri_encode_path(new_file_path)

                # 构建目标URL
                destination_url = f"{base_webdav}{encoded_new_path}"

                self.logger.debug(f"备用编码 - 源URL: {source_url}")
                self.logger.debug(f"备用编码 - 目标URL: {destination_url}")

                # 关键修复：修改Overwrite为'T'，确保覆盖目标文件
                headers = {
                    'Destination': destination_url,
                    'Overwrite': 'T'  # 修改为'T'，确保覆盖目标文件
                }

                response = session.request('MOVE', source_url, headers=headers, timeout=30)

                if response.status_code in (200, 201, 204):
                    self.logger.info(f"文件重命名成功（使用备用编码）: {webdav_path} -> {new_file_name}")
                    return
                else:
                    raise Exception(f"重命名失败，服务器返回状态码: {response.status_code}")

            except Exception as e2:
                self.logger.error(f"备用编码方式也失败: {e2}")
                raise Exception(f"无法重命名包含特殊字符的文件: {str(e2)}")

        except Exception as e:
            self.logger.error(f"重命名文件时出错: {e}")
            raise

    def delete_selected_files(self):
        """删除选中的文件"""
        self.logger.debug("删除选中的文件")
        try:
            selected_items = self.main_window.file_list.selectedItems()
            if not selected_items:
                self.logger.debug("没有选中文件")
                return
            # 确认删除
            reply = QMessageBox.question(
                self.main_window, "确认删除",
                f"确定要删除选中的 {len(selected_items)} 个文件吗?\n此操作不可恢复!",
                QMessageBox.Yes | QMessageBox.No
            )
            if reply != QMessageBox.Yes:
                self.logger.debug("用户取消了删除")
                return
            self.logger.debug(f"用户确认删除 {len(selected_items)} 个文件")
            # 删除文件
            success_count = 0
            failure_count = 0
            for item in selected_items:
                # 获取文件信息
                file_name = item.data(Qt.UserRole + 1)
                file_path = item.data(Qt.UserRole)
                try:
                    # 删除文件
                    self._delete_single_file(file_path)
                    success_count += 1
                    self.logger.info(f"文件删除成功: {file_name}")
                except Exception as e:
                    failure_count += 1
                    self.logger.debug(f"文件删除失败: {file_name}, 错误: {e}")
            # 显示结果
            if failure_count == 0:
                self.show_temp_status_message.emit(f"成功删除 {success_count} 个文件")
            else:
                self.show_temp_status_message.emit(f"删除完成: 成功 {success_count} 个, 失败 {failure_count} 个")
            # 刷新文件列表
            self.main_window.connect_and_load()
        except Exception as e:
            self.logger.debug(f"删除文件时出错: {e}")
            import traceback
            traceback.print_exc()
            QMessageBox.critical(self.main_window, "错误", f"删除文件失败: {str(e)}")

    def _delete_single_file(self, webdav_path):
        """删除单个文件"""
        self.logger.debug(f"删除单个文件: {webdav_path}")
        # 转换为相对路径
        relative_file_path = self.main_window._convert_to_relative_path(webdav_path)
        # 构建删除URL
        delete_url = self.playlist_manager.build_webdav_url(relative_file_path, include_auth=False)
        self.logger.debug(f"删除URL: {delete_url}")
        # 确保会话已初始化
        session = self.get_session()
        if session is None:
            raise Exception("无法获取HTTP会话")
        # 删除文件
        response = session.delete(delete_url, timeout=30)
        if response.status_code not in (200, 201, 204, 404):  # 404表示文件不存在，也算删除成功
            self.logger.debug(f"删除失败，状态码: {response.status_code}")
            raise Exception(f"删除失败，服务器返回状态码: {response.status_code}")
        self.logger.info(f"文件删除成功: {webdav_path}")

    def _on_upload_progress(self, current, total):
        """处理上传进度信号"""
        self.logger.info(f"上传进度: {current}/{total}")
        # 这里可以更新进度条或其他UI元素

    def _on_download_progress(self, current, total):
        """处理下载进度信号"""
        self.logger.info(f"下载进度: {current}/{total}")
        # 这里可以更新进度条或其他UI元素

    def _on_show_temp_status_message(self, message):
        """处理临时状态消息信号"""
        self.logger.debug(f"处理临时状态消息: {message}")
        try:
            # 使用状态栏显示临时消息
            self.main_window.status_label.setText(message)
            # 3秒后恢复状态栏
            from PyQt5.QtCore import QTimer
            QTimer.singleShot(3000,
                              lambda: self.main_window.status_label.setText(f"已连接: {self.main_window.webdav_url}"))
            self.logger.debug(f"临时消息已显示，将在3秒后恢复")
        except Exception as e:
            self.logger.debug(f"处理临时状态消息时出错: {e}")
            import traceback
            traceback.print_exc()

    def stop_all_threads(self):
        """停止所有后台线程"""
        self.logger.debug("停止所有后台线程")
        try:
            # 停止其他线程
            for thread in self.threads:
                if thread.is_alive():
                    self.logger.debug(f"停止线程: {thread}")
                    # 如果线程有stop方法，调用它
                    if hasattr(thread, 'stop'):
                        thread.stop()
                    # 等待线程结束
                    thread.join(timeout=1.0)
            self.logger.debug("所有线程已停止")
        except Exception as e:
            self.logger.debug(f"停止线程时出错: {e}")
            import traceback
            traceback.print_exc()


class DragDropListWidget(QListWidget):
    """支持拖拽上传的自定义列表组件"""

    def __init__(self, parent=None, main_window=None, file_manager=None):
        super().__init__(parent)
        self.parent = parent
        self.main_window = main_window
        self.file_manager = file_manager
        self.setAcceptDrops(True)
        self.setDragDropMode(QListWidget.DropOnly)
        self.setDefaultDropAction(Qt.CopyAction)

    def dragEnterEvent(self, event):
        """拖拽进入事件"""
        self.file_manager.logger.debug("拖拽进入事件")
        # 检查拖拽的数据是否包含文件URL
        if event.mimeData().hasUrls():
            # 检查URL是否指向本地文件
            urls = event.mimeData().urls()
            for url in urls:
                if url.isLocalFile():
                    event.acceptProposedAction()
                    self.file_manager.logger.debug("接受拖拽")
                    return
        self.file_manager.logger.debug("拒绝拖拽")
        event.ignore()

    def dragMoveEvent(self, event):
        """拖拽移动事件"""
        # 如果在dragEnterEvent中已经接受了，这里也接受
        if event.mimeData().hasUrls():
            event.acceptProposedAction()
        else:
            event.ignore()

    def dropEvent(self, event):
        """拖拽放下事件"""
        self.file_manager.logger.debug("拖拽放下事件")
        # 检查是否有文件URL
        if event.mimeData().hasUrls():
            urls = event.mimeData().urls()
            file_paths = []
            # 提取所有本地文件路径
            for url in urls:
                if url.isLocalFile():
                    file_path = url.toLocalFile()
                    self.file_manager.logger.debug(f"拖拽文件: {file_path}")
                    file_paths.append(file_path)
            # 如果有文件，调用上传方法
            if file_paths:
                self.file_manager.logger.debug(f"准备上传 {len(file_paths)} 个文件")
                # 调用上传方法
                self.file_manager._upload_files_with_paths(file_paths)
                event.acceptProposedAction()
                self.file_manager.logger.debug("拖拽上传已启动")
                return
        self.file_manager.logger.debug("拖拽上传未启动")
        event.ignore()
