import os
import re
import time
from urllib.parse import quote, unquote
from PyQt5.QtCore import QTimer, pyqtSignal, QObject, QUrl
from PyQt5.QtWidgets import QMessageBox, QApplication
from PyQt5.QtGui import QTextCursor

from yc import LyricDialog,LyricSyncWorker,LogManager


class LyricManager(QObject):
    """歌词管理类 - 负责所有歌词相关功能"""
    # 定义信号
    lyric_sync_progress = pyqtSignal(str)  # 同步进度信号
    lyric_sync_complete = pyqtSignal(str, bool)  # 同步完成信号 (消息, 是否成功)
    lyric_sync_error = pyqtSignal(str)  # 同步错误信号
    show_temp_status_message = pyqtSignal(str)  # 临时状态消息信号

    def __init__(self, parent):
        super().__init__(parent)
        self.parent = parent  # 主窗口引用
        self.logger = LogManager()
        # 初始化歌词相关变量
        self.lyrics = []
        self.current_lyric_index = 0
        self.lyric_file = None
        self.lyric_offset = 0  # 单位：秒
        self.lyric_sync_thread = None
        # 从主窗口获取UI组件引用
        self.lyric_display = parent.lyric_display
        self.lyric_highlighter = parent.lyric_highlighter
        self.lyric_title_label = parent.lyric_title_label
        self.current_lyric_label = parent.current_lyric_label
        self.lyric_offset_slider = parent.lyric_offset_slider
        self.lyric_offset_label = parent.lyric_offset_label
        self.logger.debug("LyricManager init ...")

        # 连接信号
        self._connect_signals()

    def _connect_signals(self):
        """连接信号"""
        self.lyric_sync_progress.connect(self._on_lyric_sync_progress)
        self.lyric_sync_complete.connect(self._on_lyric_sync_complete)
        self.lyric_sync_error.connect(self._on_lyric_sync_error)

    # ===== 属性访问器 =====
    @property
    def webdav_url(self):
        """获取WebDAV URL"""
        return self.parent.webdav_url

    @property
    def username(self):
        """获取用户名"""
        return self.parent.username

    @property
    def password(self):
        """获取密码"""
        return self.parent.password

    @property
    def asr_server_url(self):
        """获取ASR服务器URL"""
        return self.parent.asr_server_url

    @property
    def asr_api_key(self):
        """获取ASR API密钥"""
        return self.parent.asr_api_key

    @property
    def current_track(self):
        """获取当前播放曲目"""
        return self.parent.current_track

    @property
    def track_list(self):
        """获取曲目列表"""
        return self.parent.track_list

    @property
    def playing(self):
        """获取播放状态"""
        return self.parent.playing

    @property
    def paused(self):
        """获取暂停状态"""
        return self.parent.paused

    @property
    def music_player(self):
        """获取音乐播放器"""
        return self.parent.music_player

    # ===== 歌词加载功能 =====
    def load_lyric_for_song(self, song_name, song_path):
        """为歌曲加载歌词"""
        self.logger.debug(f"为歌曲加载歌词: {song_name}")
        try:
            # 清空之前的歌词
            self.lyrics = []
            self.current_lyric_index = 0
            self.lyric_file = None
            self.lyric_display.clear()
            self.current_lyric_label.setText("")  # 清空当前歌词显示
            # 尝试从WebDAV服务器加载歌词文件
            self.load_lyric_from_webdav(song_name, song_path)
            # 如果WebDAV加载失败，尝试本地加载
            if not self.lyrics:
                self.load_lyric_from_local(song_name)
            # 更新歌词标题
            if self.lyrics:
                self.lyric_title_label.setText(f"歌词 - {song_name}")
                self.show_temp_status_message.emit(f"已加载歌词: {len(self.lyrics)}行")
                # 如果正在播放，立即显示第一行歌词
                if self.playing and not self.paused:
                    self.current_lyric_label.setText(self.lyrics[0]['text'])
                return True
            else:
                self.lyric_title_label.setText(f"歌词 - {song_name} (未找到)")
                self.lyric_display.setText("未找到歌词文件\n\n您可以手动加载歌词文件")
                self.current_lyric_label.setText("无歌词")  # 显示无歌词状态
                return False
        except Exception as e:
            self.logger.debug(f"加载歌词时出错: {e}")
            import traceback
            traceback.print_exc()
            return False

    def load_lyric_from_webdav(self, song_name, song_path):
        """从WebDAV服务器加载歌词文件"""
        self.logger.debug(f"从WebDAV加载歌词: {song_name}")
        try:
            # 构建歌词文件路径（与音乐文件同名，扩展名为.lrc或.LRC）
            protocol = self.webdav_url.split('://')[0]
            host = self.webdav_url.split('://')[1].split('/')[0]
            base_webdav = f"{protocol}://{host}"
            # 获取歌曲文件名（不含扩展名）
            song_filename = os.path.splitext(song_name)[0]
            # 获取歌曲文件所在目录
            song_dir = os.path.dirname(song_path)
            lyric_extensions = ['.lrc', '.LRC']
            for ext in lyric_extensions:
                # 构建歌词URL
                lyric_url = f"{base_webdav}/{song_dir}/{song_filename}{ext}"
                self.logger.debug(f"尝试歌词URL: {lyric_url}")
                # 确保会话已初始化
                session = self.parent.get_session()
                # 请求歌词文件
                try:
                    response = session.get(lyric_url, timeout=10)
                    self.logger.debug(f"歌词文件响应状态码: {response.status_code}")
                    if response.status_code == 200:
                        # 获取原始字节数据
                        lyric_data = response.content
                        self.logger.debug(f"从WebDAV获取到歌词数据，长度: {len(lyric_data)}")
                        # 尝试不同编码解析歌词
                        lyric_content = self.decode_lyric_data(lyric_data)
                        if lyric_content is not None:
                            self.parse_lyric_content(lyric_content)
                            self.lyric_file = f"{song_filename}{ext}"
                            self.logger.debug(f"歌词加载成功: {self.lyric_file}")
                            return  # 成功加载，退出循环
                        else:
                            self.logger.debug(f"无法解码歌词文件内容: {lyric_url}")
                    else:
                        self.logger.debug(f"WebDAV歌词文件不存在，状态码: {response.status_code}")
                except Exception as e:
                    self.logger.debug(f"请求歌词文件时出错: {e}")
                    # 继续尝试下一个扩展名
            self.logger.debug("所有WebDAV歌词文件尝试均失败")
        except Exception as e:
            self.logger.debug(f"从WebDAV加载歌词失败: {e}")
            import traceback
            traceback.print_exc()

    def load_lyric_from_local(self, song_name):
        """从本地加载歌词文件"""
        self.logger.debug(f"从本地加载歌词: {song_name}")
        try:
            # 获取歌曲文件名（不含扩展名）
            song_basename = os.path.splitext(song_name)[0]
            # 尝试多个可能的歌词文件路径和扩展名
            possible_paths = []
            lyric_extensions = ['.lrc', '.LRC']
            # 优先从lrc文件夹加载
            lrc_dir = os.path.join(os.path.dirname(os.path.abspath(__file__)), "lrc")
            if not os.path.exists(lrc_dir):
                os.makedirs(lrc_dir)
                self.logger.debug(f"创建lrc文件夹: {lrc_dir}")
            for ext in lyric_extensions:
                possible_paths.append(os.path.join(lrc_dir, f"{song_basename}{ext}"))
            # 程序目录下的lyrics文件夹
            for ext in lyric_extensions:
                possible_paths.append(
                    os.path.join(os.path.dirname(os.path.abspath(__file__)), "lyrics", f"{song_basename}{ext}")
                )
            # 与音乐文件同目录
            for ext in lyric_extensions:
                possible_paths.append(
                    os.path.join(os.path.dirname(os.path.abspath(__file__)), f"{song_basename}{ext}")
                )
            # 用户文档目录
            for ext in lyric_extensions:
                possible_paths.append(
                    os.path.join(os.path.expanduser("~"), "Documents", "WebDAV Lyrics", f"{song_basename}{ext}")
                )
            # 尝试所有可能的路径
            for path in possible_paths:
                if os.path.exists(path):
                    self.logger.debug(f"找到本地歌词文件: {path}")
                    # 读取原始字节数据
                    with open(path, 'rb') as f:
                        lyric_data = f.read()
                    # 尝试不同编码解析歌词
                    lyric_content = self.decode_lyric_data(lyric_data)
                    if lyric_content is not None:
                        self.parse_lyric_content(lyric_content)
                        self.lyric_file = path
                        return
            self.logger.debug("未找到本地歌词文件")
        except Exception as e:
            self.logger.debug(f"从本地加载歌词失败: {e}")

    def decode_lyric_data(self, data):
        """尝试使用不同编码解码歌词数据"""
        # 常见的中文编码，按优先级排序
        encodings = [
            'utf-8',  # UTF-8 是最常用的编码
            'utf-8-sig',  # UTF-8 with BOM
            'gbk',  # GBK 是中文Windows常用的编码
            'gb2312',  # GB2312 是GBK的子集
            'gb18030',  # GB18030 是GBK的超集
            'big5',  # Big5 是繁体中文常用的编码
            'iso-8859-1',  # ISO-8859-1 是Latin-1编码
            'cp1252',  # Windows-1252 编码
        ]
        # 首先检查是否有BOM（字节顺序标记）
        if data.startswith(b'\xef\xbb\xbf'):
            # UTF-8 with BOM
            try:
                lyric_content = data[3:].decode('utf-8')
                self.logger.debug("检测到UTF-8 BOM，使用UTF-8解码")
                return lyric_content
            except Exception as e:
                self.logger.debug(f"UTF-8 BOM解码失败: {e}")
        elif data.startswith(b'\xff\xfe'):
            # UTF-16 Little Endian
            try:
                lyric_content = data[2:].decode('utf-16le')
                self.logger.debug("检测到UTF-16 LE BOM，使用UTF-16 LE解码")
                return lyric_content
            except Exception as e:
                self.logger.debug(f"UTF-16 LE BOM解码失败: {e}")
        elif data.startswith(b'\xfe\xff'):
            # UTF-16 Big Endian
            try:
                lyric_content = data[2:].decode('utf-16be')
                self.logger.debug("检测到UTF-16 BE BOM，使用UTF-16 BE解码")
                return lyric_content
            except Exception as e:
                self.logger.debug(f"UTF-16 BE BOM解码失败: {e}")
        # 如果没有BOM，尝试各种编码
        for encoding in encodings:
            try:
                lyric_content = data.decode(encoding)
                # 简单验证解码结果是否合理（包含中文字符）
                if any('\u4e00' <= char <= '\u9fff' for char in lyric_content):
                    self.logger.debug(f"成功使用 {encoding} 编码解码歌词（包含中文）")
                    return lyric_content
                # 如果不包含中文，但解码成功，也返回
                self.logger.debug(f"成功使用 {encoding} 编码解码歌词（不包含中文）")
                return lyric_content
            except UnicodeDecodeError:
                continue
            except Exception as e:
                self.logger.debug(f"使用 {encoding} 编码解码时出错: {e}")
                continue
        # 如果所有编码都失败，尝试使用chardet检测编码（如果可用）
        try:
            import chardet
            detection_result = chardet.detect(data)
            detected_encoding = detection_result['encoding']
            confidence = detection_result['confidence']
            if detected_encoding and confidence > 0.7:
                self.logger.debug(f"chardet检测到编码: {detected_encoding} (置信度: {confidence:.2f})")
                try:
                    lyric_content = data.decode(detected_encoding)
                    self.logger.debug("使用chardet检测的编码成功解码歌词")
                    return lyric_content
                except Exception as e:
                    self.logger.debug(f"使用chardet检测的编码解码失败: {e}")
        except ImportError:
            self.logger.debug("chardet模块未安装，跳过自动编码检测")
        except Exception as e:
            self.logger.debug(f"chardet编码检测失败: {e}")
        # 最后尝试：使用UTF-8并忽略错误
        try:
            lyric_content = data.decode('utf-8', errors='ignore')
            self.logger.debug("使用UTF-8编码（忽略错误）解码歌词")
            return lyric_content
        except Exception as e:
            self.logger.debug("所有编码尝试失败: {e}")
            return None

    def parse_lyric_content(self, content):
        """解析歌词内容"""
        self.logger.debug("解析歌词内容")
        try:
            # 清空之前的歌词
            self.lyrics = []
            # 解析LRC格式歌词
            # LRC格式: [mm:ss.xx]歌词内容 或 [mm:ss]歌词内容
            pattern = re.compile(r'\[(\d{1,2}):(\d{1,2})(?:\.(\d{1,2}))?\](.*)')
            lines = content.split('\n')
            for line in lines:
                line = line.strip()
                if not line:
                    continue
                match = pattern.match(line)
                if match:
                    minutes = int(match.group(1))
                    seconds = int(match.group(2))
                    # 处理毫秒部分（如果有）
                    if match.group(3):
                        centiseconds = int(match.group(3))
                    else:
                        centiseconds = 0
                    text = match.group(4).strip()
                    # 计算总毫秒数
                    total_ms = (minutes * 60 + seconds) * 1000 + centiseconds * 10
                    self.lyrics.append({
                        'time': total_ms,
                        'text': text,
                        'raw_line': line  # 保存原始行
                    })
                else:
                    # 对于不符合LRC格式的行，作为纯文本处理
                    self.logger.debug(f"非LRC格式行: {line}")
                    # 尝试从上一行获取时间戳，如果没有则使用0
                    last_time = self.lyrics[-1]['time'] if self.lyrics else 0
                    self.lyrics.append({
                        'time': last_time,
                        'text': line,
                        'raw_line': line
                    })
            # 按时间排序
            self.lyrics.sort(key=lambda x: x['time'])
            self.logger.debug(f"解析完成，共 {len(self.lyrics)} 行歌词")
            # 显示歌词
            self.display_all_lyrics()
        except Exception as e:
            self.logger.debug(f"解析歌词内容失败: {e}")
            import traceback
            traceback.print_exc()

    def display_all_lyrics(self):
        """显示所有歌词"""
        self.logger.debug("显示所有歌词")
        try:
            # 优化：使用setUpdatesEnabled减少UI重绘
            self.lyric_display.setUpdatesEnabled(False)
            try:
                self.lyric_display.clear()
                # 添加所有歌词，每行歌词后跟一个空行
                for lyric in self.lyrics:
                    # 优先使用原始行，如果没有则重新构建
                    if 'raw_line' in lyric:
                        self.lyric_display.append(lyric['raw_line'])
                    else:
                        minutes = lyric['time'] // 60000
                        seconds = (lyric['time'] % 60000) // 1000
                        milliseconds = (lyric['time'] % 1000) // 10
                        formatted_line = f"[{minutes:02d}:{seconds:02d}.{milliseconds:02d}]{lyric['text']}"
                        self.lyric_display.append(formatted_line)
                    self.lyric_display.append("")  # 添加空行
                # 滚动到顶部
                cursor = self.lyric_display.textCursor()
                cursor.setPosition(0)
                self.lyric_display.setTextCursor(cursor)
                # 重置当前歌词索引
                self.current_lyric_index = 0
                # 更新高亮（高亮第一行歌词）
                self.lyric_highlighter.set_current_line(0)  # 第一行歌词在文本编辑器的第0行
            finally:
                self.lyric_display.setUpdatesEnabled(True)
                # 强制更新UI
                self.lyric_display.update()
                # 确保UI立即刷新
                QApplication.processEvents()
        except Exception as e:
            self.logger.debug(f"显示所有歌词失败: {e}")
            import traceback
            traceback.print_exc()

    # ===== 歌词显示更新功能 =====
    def update_lyric_display(self, current_time_ms):
        """更新歌词显示"""
        if not self.lyrics:
            return
        try:
            # 添加容错范围（±100ms）
            tolerance = 100
            # 查找当前时间对应的歌词行
            new_index = -1
            for i, lyric in enumerate(self.lyrics):
                next_time = self.lyrics[i + 1]['time'] if i + 1 < len(self.lyrics) else float('inf')
                if (lyric['time'] - tolerance) <= current_time_ms < next_time:
                    new_index = i
                    break
            # 确保至少有一行歌词被高亮
            if new_index == -1 and self.lyrics:
                new_index = 0
            # 只有在索引变化时才更新UI
            if new_index != self.current_lyric_index:
                self.current_lyric_index = new_index
                # 更新当前歌词显示标签
                if 0 <= new_index < len(self.lyrics):
                    current_lyric_text = self.lyrics[new_index]['text']
                    self.current_lyric_label.setText(current_lyric_text)
                    # 更新主窗口歌词显示和高亮
                    # 计算在文本编辑器中的实际行号（每行歌词后跟一个空行）
                    actual_line = new_index * 2  # 因为每行歌词后跟一个空行
                    # 使用文档布局获取精确位置进行居中显示
                    cursor = self.lyric_display.textCursor()
                    # 移动到对应的块（行）
                    block = self.lyric_display.document().findBlockByNumber(actual_line)
                    if block.isValid():
                        # 获取当前块的矩形位置
                        document = self.lyric_display.document()
                        block_rect = document.documentLayout().blockBoundingRect(block)
                        block_y = block_rect.y()
                        # 获取窗口高度
                        window_height = self.lyric_display.height()
                        # 计算目标滚动位置，使当前歌词居中
                        # 目标位置 = 块的y坐标 - 窗口高度/2 + 块高度/2
                        target_pos = block_y - window_height / 2 + block_rect.height() / 2
                        # 设置光标位置
                        cursor.setPosition(block.position())
                        self.lyric_display.setTextCursor(cursor)
                        # 获取滚动条并设置滚动位置
                        scrollbar = self.lyric_display.verticalScrollBar()
                        if scrollbar:
                            # 确保目标位置在有效范围内
                            min_val = scrollbar.minimum()
                            max_val = scrollbar.maximum()
                            target_pos = max(min_val, min(max_val, int(target_pos)))
                            # 设置滚动位置
                            scrollbar.setValue(target_pos)
                        # 更新高亮（字体更大更粗）
                        self.lyric_highlighter.set_current_line(actual_line)
                        # 额外确保当前行可见
                        self.lyric_display.ensureCursorVisible()
                    # 更新独立歌词窗口
                    if hasattr(self.parent, 'lyric_window') and self.parent.lyric_window:
                        # 生成HTML内容（当前歌词字体更大更粗）
                        html_content = ""
                        for i, lyric in enumerate(self.lyrics):
                            if i == new_index:
                                # 当前歌词 - 高亮显示，字体更大更粗
                                html_content += f'<div id="current-lyric" style="color: #00BFFF; font-size: 22px; font-weight: bold; margin: 12px 0;">{lyric["text"]}</div>'
                            else:
                                # 其他歌词 - 普通显示
                                html_content += f'<div style="color:#4682B4; font-size: 16px; margin: 8px 0;">{lyric["text"]}</div>'
                        # 设置HTML内容
                        self.parent.lyric_window.lyric_display.setHtml(html_content)
                        # 使用QTextCursor定位到当前歌词行
                        cursor = self.parent.lyric_window.lyric_display.textCursor()
                        cursor.movePosition(QTextCursor.Start)
                        # 移动到当前歌词行（每个歌词是一个div块）
                        for i in range(new_index):
                            cursor.movePosition(QTextCursor.NextBlock)
                        # 设置光标位置
                        self.parent.lyric_window.lyric_display.setTextCursor(cursor)
                        # 获取滚动条
                        scrollbar = self.parent.lyric_window.lyric_display.verticalScrollBar()
                        if scrollbar:
                            # 获取文档对象
                            document = self.parent.lyric_window.lyric_display.document()
                            # 获取当前歌词块的矩形位置
                            current_block = cursor.block()
                            if current_block.isValid():
                                # 获取当前块在文档中的位置（y坐标）
                                block_rect = document.documentLayout().blockBoundingRect(current_block)
                                block_y = block_rect.y()
                                # 获取窗口高度
                                window_height = self.parent.lyric_window.lyric_display.height()
                                # 计算目标滚动位置，使当前歌词居中
                                # 目标位置 = 块的y坐标 - 窗口高度/2 + 块高度/2
                                target_pos = block_y - window_height / 2 + block_rect.height() / 2
                                # 确保目标位置在有效范围内
                                min_val = scrollbar.minimum()
                                max_val = scrollbar.maximum()
                                target_pos = max(min_val, min(max_val, int(target_pos)))
                                # 设置滚动位置
                                scrollbar.setValue(target_pos)
                                # 额外确保当前歌词可见
                                self.parent.lyric_window.lyric_display.ensureCursorVisible()
                else:
                    self.current_lyric_label.setText("")
                    # 更新独立歌词窗口
                    if hasattr(self.parent, 'lyric_window') and self.parent.lyric_window:
                        self.parent.lyric_window.lyric_display.clear()
                # 调试信息
                if new_index < len(self.lyrics):
                    self.logger.debug(
                        f"[LYRIC] 歌词索引: {new_index} | 时间: {self.lyrics[new_index]['time']}ms | 当前: {current_time_ms}ms")
        except Exception as e:
            self.logger.debug(f"更新歌词显示失败: {e}")
            import traceback
            traceback.print_exc()

    # ===== 歌词文件操作功能 =====
    def load_lyric_file(self):
        """加载歌词文件"""
        self.logger.debug("加载歌词文件")
        try:
            # 支持多种歌词文件扩展名
            file_filter = "歌词文件 (*.lrc *.LRC);;所有文件 (*)"
            file_path, _ = self.parent.QFileDialog.getOpenFileName(
                self.parent, "选择歌词文件", "", file_filter
            )
            if file_path:
                self.logger.info(f"用户选择歌词文件: {file_path}")
                # 读取原始字节数据
                with open(file_path, 'rb') as f:
                    lyric_data = f.read()
                # 尝试不同编码解析歌词
                lyric_content = self.decode_lyric_data(lyric_data)
                if lyric_content is not None:
                    self.parse_lyric_content(lyric_content)
                    self.lyric_file = file_path
                    # 更新歌词标题
                    if self.current_track:
                        self.lyric_title_label.setText(f"歌词 - {self.current_track}")
                    self.show_temp_status_message.emit(f"已加载歌词: {len(self.lyrics)}行")
                else:
                    QMessageBox.warning(self.parent, "警告", "无法解码歌词文件，请尝试其他编码的歌词文件")
        except Exception as e:
            self.logger.debug(f"加载歌词文件失败: {e}")
            QMessageBox.critical(self.parent, "错误", f"加载歌词文件失败: {str(e)}")

    def edit_lyric(self):
        """编辑歌词"""
        self.logger.info("编辑歌词")
        try:
            # 获取当前歌词内容，保留原始格式
            if self.lyrics:
                lyric_content = ""
                for lyric in self.lyrics:
                    if 'raw_line' in lyric:
                        lyric_content += lyric['raw_line'] + "\n"
                    else:
                        minutes = lyric['time'] // 60000
                        seconds = (lyric['time'] % 60000) // 1000
                        milliseconds = (lyric['time'] % 1000) // 10
                        lyric_content += f"[{minutes:02d}:{seconds:02d}.{milliseconds:02d}]{lyric['text']}\n"
            else:
                lyric_content = ""
            # 检查WebDAV写权限
            enable_upload = self.parent.check_webdav_write_permission()
            if not enable_upload:
                self.logger.debug("用户没有WebDAV写权限")
                self.show_temp_status_message.emit("提示: 您没有WebDAV写权限，歌词将仅保存到本地")
            # 显示编辑对话框
            dialog = LyricDialog(self.parent, lyric_content, enable_upload=enable_upload)
            if dialog.exec_() == self.parent.QDialog.Accepted:
                new_lyric_content = dialog.get_saved_content()
                self.logger.debug("用户保存歌词编辑")
                # 解析新的歌词内容
                self.parse_lyric_content(new_lyric_content)
                # 本地保存
                if self.lyric_file:
                    try:
                        with open(self.lyric_file, 'w', encoding='utf-8') as f:
                            f.write(new_lyric_content)
                        self.logger.info(f"歌词已保存到本地文件: {self.lyric_file}")
                        local_saved = True
                    except Exception as e:
                        self.logger.debug(f"保存本地文件失败: {e}")
                        local_saved = False
                else:
                    local_saved = False
                # WebDAV上传
                webdav_saved = False
                if dialog.should_upload_to_webdav() and self.current_track:
                    try:
                        # 显示上传状态
                        self.show_temp_status_message.emit("正在上传歌词...")
                        # 上传歌词
                        uploaded_filename = self.upload_lyric_to_webdav(self.current_track, new_lyric_content)
                        self.logger.info(f"歌词已上传到WebDAV: {uploaded_filename}")
                        webdav_saved = True
                        # 更新歌词文件路径为完整的WebDAV路径
                        self.lyric_file = uploaded_filename
                        # 显示上传成功状态
                        self.show_temp_status_message.emit("歌词上传成功")
                        QTimer.singleShot(3000, lambda: self.parent.status_label.setText(""))
                    except Exception as e:
                        self.logger.debug(f"WebDAV上传失败: {e}")
                        self.show_temp_status_message.emit("上传失败")
                        QTimer.singleShot(3000, lambda: self.parent.status_label.setText(""))
                        QMessageBox.warning(self.parent, "上传失败",
                                            f"歌词已保存到本地，但上传到WebDAV失败:\n{str(e)}")
                # 显示保存结果
                if local_saved and webdav_saved:
                    self.show_temp_status_message.emit("歌词已保存到本地并上传到WebDAV")
                elif local_saved:
                    self.show_temp_status_message.emit("歌词已保存到本地")
                elif webdav_saved:
                    self.show_temp_status_message.emit("歌词已上传到WebDAV")
                else:
                    self.show_temp_status_message.emit("歌词已更新（未保存到文件）")
        except Exception as e:
            self.logger.debug(f"编辑歌词失败: {e}")
            QMessageBox.critical(self.parent, "错误", f"编辑歌词失败: {str(e)}")

    def save_lyric(self):
        """保存歌词"""
        self.logger.debug("保存歌词")
        try:
            if not self.lyrics:
                QMessageBox.warning(self.parent, "警告", "没有歌词可保存")
                return
            # 构建歌词内容，保留原始格式
            lyric_content = ""
            for lyric in self.lyrics:
                if 'raw_line' in lyric:
                    lyric_content += lyric['raw_line'] + "\n"
                else:
                    minutes = lyric['time'] // 60000
                    seconds = (lyric['time'] % 60000) // 1000
                    milliseconds = (lyric['time'] % 1000) // 10
                    lyric_content += f"[{minutes:02d}:{seconds:02d}.{milliseconds:02d}]{lyric['text']}\n"
            # 选择保存路径，支持多种扩展名
            if self.current_track:
                default_name = f"{os.path.splitext(self.current_track)[0]}.lrc"
            else:
                default_name = "lyrics.lrc"
            file_filter = "歌词文件 (*.lrc);;歌词文件 (*.LRC);;文本文件 (*.txt);;所有文件 (*)"
            file_path, selected_filter = self.parent.QFileDialog.getSaveFileName(
                self.parent, "保存歌词文件", default_name, file_filter
            )
            if file_path:
                # 确保目录存在
                dir_path = os.path.dirname(file_path)
                if not os.path.exists(dir_path):
                    self.logger.debug(f"创建目录: {dir_path}")
                    os.makedirs(dir_path)
                # 确保文件扩展名正确
                if not (file_path.endswith('.lrc') or file_path.endswith('.LRC') or file_path.endswith('.txt')):
                    if selected_filter == "歌词文件 (*.lrc)":
                        file_path += '.lrc'
                    elif selected_filter == "歌词文件 (*.LRC)":
                        file_path += '.LRC'
                    elif selected_filter == "文本文件 (*.txt)":
                        file_path += '.txt'
                with open(file_path, 'w', encoding='utf-8') as f:
                    f.write(lyric_content)
                self.lyric_file = file_path
                self.logger.info(f"歌词已保存到文件: {file_path}")
                self.show_temp_status_message.emit("歌词已保存")
        except Exception as e:
            self.logger.debug(f"保存歌词失败: {e}")
            QMessageBox.critical(self.parent, "错误", f"保存歌词失败: {str(e)}")

    def upload_current_lyric_to_webdav(self):
        """将当前歌词上传到WebDAV"""
        self.logger.debug("上传当前歌词到WebDAV")
        if not self.lyrics:
            QMessageBox.warning(self.parent, "警告", "没有歌词可上传")
            return
        if not self.current_track:
            QMessageBox.warning(self.parent, "警告", "没有正在播放的歌曲")
            return
        try:
            # 构建歌词内容，保留原始格式
            lyric_content = ""
            for lyric in self.lyrics:
                if 'raw_line' in lyric:
                    lyric_content += lyric['raw_line'] + "\n"
                else:
                    minutes = lyric['time'] // 60000
                    seconds = (lyric['time'] % 60000) // 1000
                    milliseconds = (lyric['time'] % 1000) // 10
                    lyric_content += f"[{minutes:02d}:{seconds:02d}.{milliseconds:02d}]{lyric['text']}\n"
            # 显示上传状态
            self.show_temp_status_message.emit("正在上传歌词...")
            # 上传歌词
            uploaded_filename = self.upload_lyric_to_webdav(self.current_track, lyric_content)
            self.logger.info(f"歌词已上传到WebDAV: {uploaded_filename}")
            # 更新歌词文件路径为完整的WebDAV路径
            self.lyric_file = uploaded_filename
            # 显示上传成功状态
            self.show_temp_status_message.emit("歌词上传成功...")
            QTimer.singleShot(3000, lambda: self.parent.status_label.setText(""))
            # 显示成功消息
            self.show_temp_status_message.emit(f"歌词已上传到WebDAV: {uploaded_filename}")
        except Exception as e:
            self.logger.debug(f"上传失败: {e}")
            self.show_temp_status_message.emit("上传失败...")
            QTimer.singleShot(3000, lambda: self.parent.status_label.setText(""))
            QMessageBox.critical(self.parent, "错误", f"上传到WebDAV失败:\n{str(e)}")

    def upload_lyric_to_webdav(self, song_name, lyric_content):
        """上传歌词到WebDAV服务器"""
        self.logger.debug(f"上传歌词到WebDAV: {song_name}")
        try:
            # 获取歌曲文件名（不含扩展名）
            song_filename = os.path.splitext(song_name)[0]
            encoded_song_filename = quote(song_filename)
            # 获取歌曲文件的实际路径
            song_path = None
            for track in self.track_list:
                if track['name'] == song_name:
                    song_path = track['path']
                    break
            if not song_path:
                raise Exception(f"找不到歌曲 {song_name} 的路径信息")
            # 转换歌曲路径为相对于WebDAV根目录的路径
            relative_song_path = self.parent._convert_to_relative_path(song_path)
            # 获取歌曲文件所在目录（相对于WebDAV根目录）
            song_dir = os.path.dirname(relative_song_path)
            self.logger.debug(f"歌曲所在目录（相对）: {song_dir}")
            # 确保会话已初始化
            session = self.parent.get_session()
            # 尝试.lrc和.LRC两种扩展名
            for ext in ['.lrc', '.LRC']:
                # 上传到与音乐文件相同的目录（相对于WebDAV根目录）
                if song_dir == '/':
                    # 如果歌曲在WebDAV根目录，直接使用文件名
                    upload_url = self.parent.playlist_manager.build_webdav_url(
                        f"/{encoded_song_filename}{ext}", include_auth=False)
                else:
                    upload_url = self.parent.playlist_manager.build_webdav_url(
                        f"{song_dir}/{encoded_song_filename}{ext}", include_auth=False)
                self.logger.debug(f"歌词上传URL: {upload_url}")
                # 上传歌词文件
                headers = {
                    'Content-Type': 'text/plain; charset=utf-8'
                }
                response = session.put(upload_url, data=lyric_content.encode('utf-8'),
                                       headers=headers, timeout=10)
                if response.status_code in (200, 201, 204):
                    self.logger.debug(f"歌词上传成功: {upload_url}")
                    # 返回相对路径，便于后续使用
                    if song_dir == '/':
                        return f"/{song_filename}{ext}"
                    else:
                        return f"{song_dir}/{song_filename}{ext}"
                else:
                    self.logger.debug(f"上传失败，状态码: {response.status_code}")
            # 两种扩展名都尝试失败
            raise Exception(f"上传失败，服务器返回状态码: {response.status_code}")
        except Exception as e:
            self.logger.debug(f"上传歌词到WebDAV失败: {e}")
            raise Exception(f"上传到WebDAV失败: {str(e)}")

    # ===== 歌词时间调整功能 =====
    def adjust_lyric_time(self):
        """调整歌词时间"""
        self.logger.info("调整歌词时间")
        try:
            if not self.lyrics:
                QMessageBox.warning(self.parent, "警告", "没有歌词可调整")
                return
            # 修改对话框提示和范围
            offset, ok = self.parent.QInputDialog.getInt(
                self.parent, "调整歌词时间",
                "请输入时间偏移（秒）:\n正数表示歌词延后，负数表示歌词提前",
                self.lyric_offset, -5, 5, 1
            )
            if ok:
                self.lyric_offset = offset
                self.lyric_offset_slider.setValue(offset)
                # 修改标签显示单位为秒
                self.lyric_offset_label.setText(f"{offset}s")
                self.logger.info(f"歌词时间偏移已设置为: {offset}秒")
                self.show_temp_status_message.emit(f"歌词时间偏移已设置为: {offset}秒")
        except Exception as e:
            self.logger.debug(f"调整歌词时间失败: {e}")
            QMessageBox.critical(self.parent, "错误", f"调整歌词时间失败: {str(e)}")

    def set_lyric_offset(self, offset):
        """设置歌词时间偏移"""
        self.logger.info(f"设置歌词时间偏移: {offset}秒")
        try:
            self.lyric_offset = offset
            # 修改标签显示单位为秒
            self.lyric_offset_label.setText(f"{offset}s")
            # 如果正在播放，立即更新歌词显示
            if self.playing and not self.paused:
                current_time_ms = self.music_player.get_current_time()
                if current_time_ms >= 0:
                    # 将秒转换为毫秒
                    self.update_lyric_display(current_time_ms + self.lyric_offset * 1000)
        except Exception as e:
            self.logger.debug(f"设置歌词时间偏移失败: {e}")
            import traceback
            traceback.print_exc()

    # ===== 歌词同步功能 =====
    def sync_lyric_online(self):
        """在线同步歌词"""
        self.logger.debug("在线同步歌词")
        # 安全地检查线程状态
        try:
            if (hasattr(self, 'lyric_sync_thread') and self.lyric_sync_thread and
                    self.lyric_sync_thread.isRunning()):
                self.show_temp_status_message.emit("歌词同步正在进行中，请稍候...")
                return
        except RuntimeError:
            # 线程对象已被删除，清理引用
            self.lyric_sync_thread = None
        # 获取当前播放歌曲的路径
        song_path = None
        for track in self.track_list:
            if track['name'] == self.current_track:
                song_path = track['path']
                break
        if not song_path:
            self.show_temp_status_message.emit("无法获取当前播放歌曲的路径")
            return
        # 创建并启动同步线程
        self.lyric_sync_thread = LyricSyncWorker(
            self.parent,
            self.current_track,
            song_path,
            self.asr_server_url,
            self.asr_api_key
        )
        # 连接信号，使用QueuedConnection确保跨线程安全
        self.lyric_sync_thread.progress.connect(
            self._on_lyric_sync_progress_update, self.parent.Qt.QueuedConnection)
        self.lyric_sync_thread.finished.connect(
            self._on_lyric_sync_complete, self.parent.Qt.QueuedConnection)
        self.lyric_sync_thread.error.connect(
            self._on_lyric_sync_error, self.parent.Qt.QueuedConnection)
        # 线程完成后清理
        self.lyric_sync_thread.finished.connect(
            self._cleanup_lyric_sync_thread, self.parent.Qt.QueuedConnection)
        self.lyric_sync_thread.error.connect(
            self._cleanup_lyric_sync_thread, self.parent.Qt.QueuedConnection)
        # 启动线程
        self.lyric_sync_thread.start()

    def _cleanup_lyric_sync_thread(self):
        """清理歌词同步线程"""
        try:
            if hasattr(self, 'lyric_sync_thread') and self.lyric_sync_thread:
                # 先断开所有信号连接
                try:
                    self.lyric_sync_thread.progress.disconnect()
                    self.lyric_sync_thread.finished.disconnect()
                    self.lyric_sync_thread.error.disconnect()
                except:
                    pass  # 忽略断开连接时的错误
                # 使用QTimer延迟删除，确保所有信号处理完成
                QTimer.singleShot(100, self._safe_delete_thread)
        except Exception as e:
            self.logger.debug(f"清理歌词同步线程时出错: {e}")
            self.lyric_sync_thread = None

    def _safe_delete_thread(self):
        """安全删除线程对象"""
        try:
            if hasattr(self, 'lyric_sync_thread') and self.lyric_sync_thread:
                self.lyric_sync_thread.deleteLater()
                self.lyric_sync_thread = None
                self.logger.debug("歌词同步线程已删除")
        except Exception as e:
            self.logger.debug(f"删除线程对象时出错: {e}")
            self.lyric_sync_thread = None




    # ===== 信号处理方法 =====
    def _on_lyric_sync_progress(self, message):
        """处理歌词同步进度信号"""
        try:
            # 使用QTimer确保在主线程中执行
            QTimer.singleShot(0, lambda: self.show_temp_status_message.emit(message))
        except Exception as e:
            self.logger.debug(f"处理歌词同步进度时发生异常: {e}")
            import traceback
            traceback.print_exc()

    def _on_lyric_sync_progress_update(self, message):
        """处理歌词同步进度信号，更新歌词标题标签"""
        try:
            if hasattr(self, 'lyric_title_label') and self.lyric_title_label:
                self.lyric_title_label.setText(f"歌词同步 - {message}")
        except Exception as e:
            self.logger.debug(f"更新歌词同步进度显示时出错: {e}")
            import traceback
            traceback.print_exc()

    def _on_lyric_sync_complete(self, message, success):
        """处理歌词同步完成信号"""
        try:
            self.show_temp_status_message.emit(message)
            self.logger.debug(f"歌词同步完成1: {message}, 成功: {success}")
            if success:
                # 如果同步成功，使用QTimer延迟执行歌词更新，确保在主线程中执行
                QTimer.singleShot(100, self._refresh_lyric_after_sync)
        except Exception as e:
            self.logger.debug(f"处理歌词同步完成时发生异常: {e}")
            import traceback
            traceback.print_exc()

    def _handle_lyric_sync_complete(self, message, success):
        """实际处理歌词同步完成的逻辑"""
        try:
            self.show_temp_status_message.emit(message)
            self.logger.debug(f"歌词同步完成: {message}, 成功: {success}")
            if success:
                # 如果同步成功，使用QTimer延迟执行歌词更新，确保在主线程中执行
                QTimer.singleShot(100, self._refresh_lyric_after_sync)
            # 同步完成后恢复歌词标题标签显示
            if hasattr(self, 'lyric_title_label') and self.lyric_title_label:
                if self.playing and self.current_track:
                    self.lyric_title_label.setText(f"歌词 - {self.current_track}")
                else:
                    self.lyric_title_label.setText("歌词显示")
        except Exception as e:
            self.logger.debug(f"处理歌词同步完成时发生异常: {e}")
            import traceback
            traceback.print_exc()

    def _refresh_lyric_after_sync(self):
        """同步完成后刷新歌词显示"""
        try:
            # 如果正在播放，重新加载歌词文件
            if self.playing and not self.paused and self.current_track:
                # 获取当前播放歌曲的路径
                song_path = None
                for track in self.track_list:
                    if track['name'] == self.current_track:
                        song_path = track['path']
                        break
                if song_path:
                    # 重新加载歌词文件
                    self.load_lyric_for_song(self.current_track, song_path)
                    self.logger.debug("歌词同步成功后重新加载歌词文件")
        except Exception as e:
            self.logger.debug(f"刷新歌词显示时出错: {e}")
            import traceback
            traceback.print_exc()

    def _on_lyric_sync_error(self, error_message):
        """处理歌词同步错误信号"""
        try:
            # 使用QTimer确保在主线程中执行
            QTimer.singleShot(0, lambda: self._handle_lyric_sync_error(error_message))
        except Exception as e:
            self.logger.debug(f"处理歌词同步错误时发生异常: {e}")
            import traceback
            traceback.print_exc()

    def _handle_lyric_sync_error(self, error_message):
        """实际处理歌词同步错误的逻辑"""
        try:
            self.show_temp_status_message.emit(error_message)
            self.logger.debug(f"歌词同步错误: {error_message}")
            # 错误发生后恢复歌词标题标签显示
            if hasattr(self, 'lyric_title_label') and self.lyric_title_label:
                if self.playing and self.current_track:
                    self.lyric_title_label.setText(f"歌词 - {self.current_track}")
                else:
                    self.lyric_title_label.setText("歌词显示")
        except Exception as e:
            self.logger.debug(f"处理歌词同步错误时发生异常: {e}")
            import traceback
            traceback.print_exc()

