# ===== 新增：日志管理模块 =====
import gzip
import inspect
import logging
import logging.handlers
import os
import shutil
import threading
import time
from datetime import datetime, timedelta


class LogManager:
    """日志管理类"""
    _instance = None
    _lock = threading.Lock()

    def __new__(cls, *args, **kwargs):
        with cls._lock:
            if cls._instance is None:
                cls._instance = super().__new__(cls)
        return cls._instance

    def __init__(self, log_dir="logs", log_level=logging.INFO,
                 max_log_size=10 * 1024 * 1024, backup_count=5,
                 retention_days=30, compress=True):
        """
        初始化日志管理器
        参数:
            log_dir: 日志目录
            log_level: 日志级别 (DEBUG, INFO, WARNING, ERROR, CRITICAL)
            max_log_size: 单个日志文件最大大小 (字节)
            backup_count: 保留的备份文件数量
            retention_days: 日志保留天数
            compress: 是否压缩旧日志文件
        """
        # 避免重复初始化
        if hasattr(self, '_initialized'):
            return
        self.log_dir = log_dir
        self.log_level = log_level
        self.max_log_size = max_log_size
        self.backup_count = backup_count
        self.retention_days = retention_days
        self.compress = compress
        # 创建日志目录
        os.makedirs(self.log_dir, exist_ok=True)
        # 设置日志格式
        self.log_format = "%(asctime)s [%(levelname)s] %(message)s"
        self.date_format = "%Y-%m-%d %H:%M:%S"
        # 创建日志记录器
        self.logger = logging.getLogger()  # 根日志器
        self.logger.setLevel(self.log_level)
        # 避免重复添加处理器
        if not self.logger.handlers:
            # 创建文件处理器
            file_handler = logging.handlers.RotatingFileHandler(
                filename=os.path.join(self.log_dir, "webdav_music_player.log"),
                maxBytes=self.max_log_size,
                backupCount=self.backup_count,
                encoding='utf-8'
            )
            file_handler.setFormatter(logging.Formatter(
                self.log_format,
                datefmt=self.date_format
            ))
            self.logger.addHandler(file_handler)
            # 创建控制台处理器
            console_handler = logging.StreamHandler()
            console_handler.setFormatter(logging.Formatter(
                self.log_format,
                datefmt=self.date_format
            ))
            self.logger.addHandler(console_handler)
        # 启动日志清理任务
        self.start_log_cleanup()
        self._initialized = True

    def start_log_cleanup(self):
        """启动日志清理任务"""
        self._cleanup_event = threading.Event()

        def cleanup_task():
            while not self._cleanup_event.is_set():
                time.sleep(24 * 60 * 60)  # 每天执行一次
                if not self._cleanup_event.is_set():
                    self.cleanup_old_logs()

        self.cleanup_thread = threading.Thread(target=cleanup_task, daemon=True)
        self.cleanup_thread.start()

    def stop_cleanup(self):
        """停止日志清理任务"""
        if hasattr(self, '_cleanup_event'):
            self._cleanup_event.set()
            if hasattr(self, 'cleanup_thread') and self.cleanup_thread.is_alive():
                self.cleanup_thread.join()

    def cleanup_old_logs(self):
        """清理旧的日志文件"""
        try:
            now = datetime.now()
            cutoff_date = now - timedelta(days=self.retention_days)
            for filename in os.listdir(self.log_dir):
                file_path = os.path.join(self.log_dir, filename)
                # 跳过当前日志文件
                if filename == "webdav_music_player.log":
                    continue
                # 检查文件修改时间
                file_mtime = datetime.fromtimestamp(os.path.getmtime(file_path))
                if file_mtime < cutoff_date:
                    try:
                        os.remove(file_path)
                        self.logger.info(f"已删除旧日志文件: {filename}")
                    except Exception as e:
                        self.logger.error(f"删除日志文件失败: {filename}, 错误: {e}")
        except Exception as e:
            self.logger.error(f"日志清理任务失败: {e}")

    def compress_log_file(self, file_path):
        """压缩日志文件"""
        try:
            gzip_path = file_path + '.gz'
            with open(file_path, 'rb') as f_in:
                with gzip.open(gzip_path, 'wb') as f_out:
                    shutil.copyfileobj(f_in, f_out)
            os.remove(file_path)
            return True
        except Exception as e:
            self.logger.error(f"压缩日志文件失败: {file_path}, 错误: {e}")
            return False

    def debug(self, message):
        """记录调试信息"""
        frame = inspect.currentframe().f_back
        filename = os.path.basename(frame.f_code.co_filename)
        lineno = frame.f_lineno
        self.logger.debug(f"{filename}:{lineno} - {message}")

    def info(self, message):
        """记录一般信息"""
        frame = inspect.currentframe().f_back
        filename = os.path.basename(frame.f_code.co_filename)
        lineno = frame.f_lineno
        self.logger.info(f"{filename}:{lineno} - {message}")

    def warning(self, message):
        """记录警告信息"""
        frame = inspect.currentframe().f_back
        filename = os.path.basename(frame.f_code.co_filename)
        lineno = frame.f_lineno
        self.logger.warning(f"{filename}:{lineno} - {message}")

    def error(self, message):
        """记录错误信息"""
        frame = inspect.currentframe().f_back
        filename = os.path.basename(frame.f_code.co_filename)
        lineno = frame.f_lineno
        self.logger.error(f"{filename}:{lineno} - {message}")

    def critical(self, message):
        """记录严重错误信息"""
        frame = inspect.currentframe().f_back
        filename = os.path.basename(frame.f_code.co_filename)
        lineno = frame.f_lineno
        self.logger.critical(f"{filename}:{lineno} - {message}")

    def set_level(self, level):
        """设置日志级别"""
        self.logger.setLevel(level)
        # 同步更新所有处理器的级别
        for handler in self.logger.handlers:
            handler.setLevel(level)
        self.log_level = level