# ===== 新增：日志管理模块 =====
from PyQt5.QtGui import QFont, QColor, QSyntaxHighlighter, \
    QTextCharFormat
from PyQt5.QtCore import QTimer
from yc import LogManager


class LyricHighlighter(QSyntaxHighlighter):
    """歌词高亮器"""

    def __init__(self, parent, logger=None, config=None):
        super().__init__(parent)
        self.logger = logger or  LogManager()
        self.config = config or {
            'font_family': 'Arial',
            'font_size': 14,
            'font_weight': QFont.Bold,
            'background_color': QColor(135, 206, 250)  # LightSkyBlue
        }
        self.current_line_format = QTextCharFormat()
        self._setup_format()
        self.current_line = -1
        self.last_highlighted = -1
        self._highlight_timer = None

    def _setup_format(self):
        """设置高亮格式"""
        font = QFont(
            self.config['font_family'],
            self.config['font_size'],
            self.config['font_weight']
        )
        self.current_line_format.setFont(font)

        # 1. 找到主窗口
        main_window = self._find_main_window()

        # 2. 获取并解析CSS
        css = main_window.styleSheet()
        import re
        styles = re.search(r'#current-lyric-label[^{]*\{([^}]+)\}', css).group(1)

        # 3. 提取颜色
        color = re.search(r'color:\s*([^;]+)', styles).group(1)
        bg_color = re.search(r'background-color:\s*([^;]+)', styles).group(1)

        # 4. 应用颜色
        self.current_line_format.setForeground(self._parse_color(color))
        self.current_line_format.setBackground(self._parse_color(bg_color))

        self.current_line_format.setFontWeight(self.config['font_weight'])

    def _find_main_window(self):
        """通过全局应用实例查找主窗口"""
        from PyQt5.QtWidgets import QApplication
        app = QApplication.instance()
        for widget in app.topLevelWidgets():
            if widget.objectName() == 'maincs':
                return widget
        # 如果找不到，可以 raise 一个明确的错误，方便调试
        raise RuntimeError("无法找到主窗口 'maincs'")

    def _parse_color(self, color_str):
        """解析颜色字符串"""
        color_str = color_str.strip()
        if color_str.startswith('#'):
            return QColor(color_str)
        if 'rgb' in color_str:
            import re
            values = re.findall(r'[\d.]+', color_str)
            if len(values) >= 4:
                r, g, b, a = [float(v) for v in values[:4]]
                if a <= 1: a = int(a * 255)
                return QColor(int(r), int(g), int(b), int(a))
            elif len(values) >= 3:
                r, g, b = [int(float(v)) for v in values[:3]]
                return QColor(r, g, b)

    def set_current_line(self, line):
        """设置当前行（带防抖）"""
        if line != self.current_line:
            self.current_line = line
            self.logger.debug(f"设置当前高亮行: {line}")
            # 防抖处理，避免频繁重绘
            if self._highlight_timer:
                self._highlight_timer.stop()
            self._highlight_timer = QTimer()
            self._highlight_timer.setSingleShot(True)
            self._highlight_timer.timeout.connect(self._do_highlight)
            self._highlight_timer.start(100)  # 100ms延迟

    def _do_highlight(self):
        """执行高亮操作"""
        if self.current_line != self.last_highlighted:
            self.last_highlighted = self.current_line
            self.logger.debug(f"重新高亮行: {self.current_line}")
            self.rehighlight()

    def highlightBlock(self, text):
        """重写高亮块方法，应用格式到文本块"""
        # 获取当前块的行号
        current_block = self.currentBlock()
        block_number = current_block.blockNumber()
        # 如果当前块是需要高亮的行，应用格式
        if block_number == self.current_line:
            self.setFormat(0, len(text), self.current_line_format)
            self.logger.debug(f"高亮行 {block_number}")